<?php
/**
 * Database Manager - Multi-Bot Shared Database System
 * Version: 3.0
 * Author: DeepSeek
 */

class Database {
    private static $instance = null;
    private $connection;
    private $isConnected = false;

    // تنظیمات دیتابیس (در config.php تعریف میشن)
    private $dbHost;
    private $dbUser;
    private $dbPass;
    private $dbName;

    public function __construct() {
        // دریافت تنظیمات از فایل پیکربندی
        require_once __DIR__ . '/../config.php';
        
        $this->dbHost = defined('DB_HOST') ? DB_HOST : 'localhost';
        $this->dbUser = defined('DB_USER') ? DB_USER : 'root';
        $this->dbPass = defined('DB_PASS') ? DB_PASS : '';
        $this->dbName = defined('DB_NAME') ? DB_NAME : 'uploader_pro_db';

        $this->connect();
        $this->checkTables();
    }

    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function connect() {
        try {
            $this->connection = new mysqli(
                $this->dbHost,
                $this->dbUser,
                $this->dbPass,
                $this->dbName
            );

            if ($this->connection->connect_error) {
                throw new Exception("Database connection failed: " . $this->connection->connect_error);
            }

            $this->isConnected = true;
            $this->connection->set_charset("utf8mb4");
        } catch (Exception $e) {
            error_log($e->getMessage());
            $this->isConnected = false;
        }
    }

    private function checkTables() {
        if (!$this->isConnected) return;

        // ایجاد جداول اگر وجود ندارند
        $this->connection->query("
            CREATE TABLE IF NOT EXISTS `files` (
                `id` VARCHAR(40) PRIMARY KEY,
                `file_name` VARCHAR(255) NOT NULL,
                `file_path` VARCHAR(255) NOT NULL,
                `uploader_id` BIGINT NOT NULL,
                `upload_time` DATETIME DEFAULT CURRENT_TIMESTAMP,
                `download_count` INT DEFAULT 0,
                `expire_time` DATETIME DEFAULT NULL,
                INDEX `idx_uploader` (`uploader_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
        ");

        $this->connection->query("
            CREATE TABLE IF NOT EXISTS `bots` (
                `id` INT AUTO_INCREMENT PRIMARY KEY,
                `token` VARCHAR(100) NOT NULL UNIQUE,
                `status` ENUM('active','inactive','maintenance') DEFAULT 'active',
                `last_ping` DATETIME DEFAULT CURRENT_TIMESTAMP,
                `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
        ");

        $this->connection->query("
            CREATE TABLE IF NOT EXISTS `downloads` (
                `id` BIGINT AUTO_INCREMENT PRIMARY KEY,
                `file_id` VARCHAR(40) NOT NULL,
                `bot_id` INT NOT NULL,
                `user_id` BIGINT NOT NULL,
                `download_time` DATETIME DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (`file_id`) REFERENCES `files`(`id`) ON DELETE CASCADE,
                FOREIGN KEY (`bot_id`) REFERENCES `bots`(`id`) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
        ");
    }

    public function getConnection() {
        if (!$this->isConnected) {
            $this->connect(); // تلاش مجدد برای اتصال
        }
        return $this->connection;
    }

    public function isConnected() {
        return $this->isConnected;
    }

    // ====== متدهای مدیریت فایل‌ها ====== //
    
    /**
     * ذخیره اطلاعات فایل جدید در دیتابیس
     */
    public function saveFile($fileId, $fileName, $filePath, $uploaderId, $expireHours = 72) {
        $conn = $this->getConnection();
        if (!$conn) return false;

        $expireTime = date('Y-m-d H:i:s', time() + ($expireHours * 3600));
        
        $stmt = $conn->prepare("
            INSERT INTO `files` 
                (id, file_name, file_path, uploader_id, expire_time) 
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->bind_param("sssis", $fileId, $fileName, $filePath, $uploaderId, $expireTime);
        
        return $stmt->execute();
    }

    /**
     * بازیابی اطلاعات فایل بر اساس ID
     */
    public function getFileById($fileId) {
        $conn = $this->getConnection();
        if (!$conn) return null;

        $stmt = $conn->prepare("
            SELECT * FROM `files` 
            WHERE `id` = ? 
            AND (`expire_time` IS NULL OR `expire_time` > NOW())
        ");
        $stmt->bind_param("s", $fileId);
        $stmt->execute();
        
        $result = $stmt->get_result();
        return $result->fetch_assoc();
    }

    /**
     * افزایش شمارنده دانلود فایل
     */
    public function incrementDownloadCount($fileId) {
        $conn = $this->getConnection();
        if (!$conn) return false;

        return $conn->query("
            UPDATE `files` 
            SET `download_count` = `download_count` + 1 
            WHERE `id` = '$fileId'
        ");
    }

    // ====== متدهای مدیریت ربات‌ها ====== //
    
    /**
     * ثبت ربات جدید یا به‌روزرسانی وضعیت
     */
    public function registerBot($token, $status = 'active') {
        $conn = $this->getConnection();
        if (!$conn) return false;

        $stmt = $conn->prepare("
            INSERT INTO `bots` (token, status) 
            VALUES (?, ?)
            ON DUPLICATE KEY UPDATE 
                status = VALUES(status),
                last_ping = NOW()
        ");
        $stmt->bind_param("ss", $token, $status);
        
        return $stmt->execute();
    }

    /**
     * دریافت لیست ربات‌های فعال
     */
    public function getActiveBots($limit = 5) {
        $conn = $this->getConnection();
        if (!$conn) return [];

        $result = $conn->query("
            SELECT `id`, `token` 
            FROM `bots` 
            WHERE `status` = 'active' 
            ORDER BY `last_ping` DESC 
            LIMIT $limit
        ");
        
        $bots = [];
        while ($row = $result->fetch_assoc()) {
            $bots[] = $row;
        }
        return $bots;
    }

    /**
     * تغییر وضعیت ربات
     */
    public function setBotStatus($token, $status) {
        $conn = $this->getConnection();
        if (!$conn) return false;

        $stmt = $conn->prepare("
            UPDATE `bots` 
            SET `status` = ? 
            WHERE `token` = ?
        ");
        $stmt->bind_param("ss", $status, $token);
        
        return $stmt->execute();
    }

    // ====== متدهای گزارش‌گیری ====== //
    
    /**
     * ثبت دانلود جدید
     */
    public function logDownload($fileId, $botId, $userId) {
        $conn = $this->getConnection();
        if (!$conn) return false;

        $stmt = $conn->prepare("
            INSERT INTO `downloads` 
                (file_id, bot_id, user_id) 
            VALUES (?, ?, ?)
        ");
        $stmt->bind_param("sii", $fileId, $botId, $userId);
        
        return $stmt->execute();
    }

    /**
     * پاک‌سازی فایل‌های منقضی شده
     */
    public function cleanupExpiredFiles() {
        $conn = $this->getConnection();
        if (!$conn) return false;

        // اول فایل‌های منقضی را پیدا می‌کنیم
        $expiredFiles = $conn->query("
            SELECT `id`, `file_path` 
            FROM `files` 
            WHERE `expire_time` < NOW()
        ");

        $deleted = 0;
        while ($file = $expiredFiles->fetch_assoc()) {
            // حذف فایل فیزیکی
            if (file_exists($file['file_path'])) {
                unlink($file['file_path']);
            }
            
            // حذف رکورد دیتابیس
            $conn->query("DELETE FROM `files` WHERE `id` = '{$file['id']}'");
            $deleted++;
        }
        
        return $deleted;
    }

    // ====== توابع کمکی ====== //

    public function escapeString($value) {
        if (!$this->isConnected) return addslashes($value);
        return $this->connection->real_escape_string($value);
    }

    public function getLastError() {
        return $this->isConnected ? $this->connection->error : 'No database connection';
    }

    public function __destruct() {
        if ($this->isConnected) {
            $this->connection->close();
        }
    }
}
?>